// SDK:         nRF Connect SDK v3.1.1
// Toolchain:   nRF Connect SDK Toolchain v3.1.1
// seeedboards: zephyr version ~3.40201.251021
// board:       xiao_nrf54l15 sense
// 2025/11/19

#include <stdio.h>
#include <string.h>

#include <zephyr/kernel.h>
#include <zephyr/device.h>
#include <zephyr/drivers/i2c.h>
#include <zephyr/drivers/regulator.h>
#include <zephyr/drivers/gpio.h>
#include <zephyr/drivers/adc.h>
#include <zephyr/drivers/display.h>
#include <zephyr/devicetree.h>
#include <nrfx_power.h>
#include <nrf.h>

#include <zephyr/bluetooth/bluetooth.h>
#include <zephyr/bluetooth/conn.h>
#include <zephyr/bluetooth/hci.h>
#include <zephyr/bluetooth/hci_vs.h>
#include <zephyr/bluetooth/gap.h>
#include <zephyr/bluetooth/gatt.h>
#include <zephyr/bluetooth/uuid.h>

#include <zephyr/sys/printk.h>
#include <zephyr/random/random.h>
#include <zephyr/sys/timeutil.h>
#include <zephyr/sys/util.h>
#include <zephyr/sys/byteorder.h>

#include <lvgl.h>   // v9.3.0

#include <zephyr/logging/log.h>

LOG_MODULE_REGISTER(main, CONFIG_LOG_DEFAULT_LEVEL);

// **************************
// Ports and on-board devices
// **************************
static const struct gpio_dt_spec led_builtin = GPIO_DT_SPEC_GET(DT_ALIAS(led0), gpios);
static const struct gpio_dt_spec usrbtn = GPIO_DT_SPEC_GET(DT_ALIAS(sw0), gpios);
static const struct gpio_dt_spec test0 = GPIO_DT_SPEC_GET(DT_ALIAS(p104d0), gpios);
static const struct gpio_dt_spec selsw = GPIO_DT_SPEC_GET(DT_ALIAS(p105d1), gpios);
static const struct gpio_dt_spec rfswctl = GPIO_DT_SPEC_GET(DT_NODELABEL(rfsw_ctl), enable_gpios);

static const struct device *const pdm_imu_reg = DEVICE_DT_GET(DT_NODELABEL(pdm_imu_pwr));
static const struct device *const rfsw_reg = DEVICE_DT_GET(DT_NODELABEL(rfsw_pwr));
static const struct device *const vbat_reg = DEVICE_DT_GET(DT_NODELABEL(vbat_pwr));

// --------------------------------------
// Initialize function of on-board device
// --------------------------------------
int port_init(void) {
	int err;
    bool rdy;
    
    rdy = gpio_is_ready_dt(&led_builtin);	// on board green led
    if (rdy == false) { return -11; }
    rdy = gpio_is_ready_dt(&usrbtn);		// on board push sw
    if (rdy == false) { return -12; }
    rdy = gpio_is_ready_dt(&test0);		    // xiao port D0
    if (rdy == false) { return -13; }
    rdy = gpio_is_ready_dt(&selsw);			// xioa port D1
    if (rdy == false) { return -14; }
	rdy = device_is_ready(rfswctl.port);	// on board resw control
    if (rdy == false) { return -15; }

    err = gpio_pin_configure_dt(&led_builtin, GPIO_OUTPUT_ACTIVE);	// on board green led
	if (err < 0) { return -21; }
    err = gpio_pin_configure_dt(&usrbtn, GPIO_INPUT);				// on board push sw
    if (err < 0) { return -22; }
    err = gpio_pin_configure_dt(&test0, GPIO_OUTPUT_ACTIVE);		// xiao port D0
    if (err < 0) { return 23; }
    err = gpio_pin_configure_dt(&selsw, GPIO_INPUT);				// xioa port D1
    if (err < 0) { return -24; }
	err = gpio_pin_configure_dt(&rfswctl, GPIO_OUTPUT_ACTIVE);		// on board resw control
    if (err < 0) { return -25; }

	// on-board devices power
    regulator_disable(pdm_imu_reg);     // on board pdm and imu power: OFF
	regulator_enable(rfsw_reg);         // on board rfsw power: ON
	regulator_enable(vbat_reg);        	// on board battery voltage read: ON

    // antenna select
    err = gpio_pin_set_dt(&rfswctl, 1); // 1:onboard, 0:external
    if (err < 0) { return -3; }

	return 0;
}

// ************************
// battery voltage read ADC
// ************************
#define DT_SPEC_AND_COMMA(node_id, prop, idx) ADC_DT_SPEC_GET_BY_IDX(node_id, idx),

static const struct adc_dt_spec adc_channels[] = {
	DT_FOREACH_PROP_ELEM(DT_PATH(zephyr_user), io_channels, DT_SPEC_AND_COMMA)
};

static uint16_t adc_buf;
static int32_t val_mv;
static struct adc_sequence sequence = {
	.buffer = &adc_buf,
	.buffer_size = sizeof(adc_buf),
};

// --------------------------
// Initialize function of adc
// --------------------------
int adc_init(void) {
	int err;

	err = adc_is_ready_dt(&adc_channels[7]);
	if (err < 0) { return -1; }

	err = adc_channel_setup_dt(&adc_channels[7]);
	if (err < 0) { return -2; }

	err = adc_sequence_init_dt(&adc_channels[7], &sequence);
	if (err < 0) { return -3; }

	return 0;
}

// **********************
// SSD1306 Oled with lvgl
// **********************
#define FONTNAME ibmlight
//#define FONTNAME ibmregular
//#define FONTNAME dejavu15
//#define FONTNAME spacemono

static  lv_obj_t *line_1;
static  lv_obj_t *line_2;
static  lv_obj_t *line_3;
static  lv_obj_t *line_4;
static  lv_style_t style;

int ssd_init(void) {
    LV_FONT_DECLARE(FONTNAME);   // 14 characters / LINE
	const struct device *display_dev;

	display_dev = DEVICE_DT_GET(DT_CHOSEN(zephyr_display));
	if (!device_is_ready(display_dev)) {
		LOG_ERR("Device not ready, aborting test");
		return -1;
	}

    // label, font, and style definition
    lv_style_init(&style);
    lv_style_set_text_font(&style, &FONTNAME);
    
	line_1 = lv_label_create(lv_scr_act());
    lv_obj_set_pos(line_1, 0, 0);
    lv_obj_add_style(line_1, &style, 0);
	line_2 = lv_label_create(lv_scr_act());
    lv_obj_set_pos(line_2, 0, 16);
    lv_obj_add_style(line_2, &style, 0);
    line_3 = lv_label_create(lv_scr_act());
    lv_obj_set_pos(line_3, 0, 32);
    lv_obj_add_style(line_3, &style, 0);
	line_4 = lv_label_create(lv_scr_act());
    lv_obj_set_pos(line_4, 0, 48);
    lv_obj_add_style(line_4, &style, 0);

    // title display
	display_blanking_off(display_dev);  // display ON

    lv_label_set_text(line_1, " BLE");
    lv_label_set_text(line_2, "   CENTRAL");
    lv_label_set_text(line_3, "");
    lv_label_set_text(line_4, "XIAO_nRF54L15");

	lv_task_handler();
	k_msleep(1000);

	return 0;
}

// **************
// BLE Central 
// **************
#define DATANUM 16  // timestamp 4, random number 8, battery voltage 2, tx_power/rssi 2
union unionData {   // Union for data type conversion
  uint32_t  dataBuff32[DATANUM/4];
  uint16_t  dataBuff16[DATANUM/2];
  uint8_t   dataBuff8[DATANUM];
};
union unionData ud;
static uint8_t recv_buf[DATANUM];
static uint8_t send_buf[DATANUM];

static struct bt_uuid_128 service_uuid = BT_UUID_INIT_128(
	0x85, 0xb4, 0x93, 0x81, 0x58, 0x40, 0x0c, 0xbe,
	0xd1, 0x4d, 0x00, 0x00, 0xc4, 0x55, 0x00, 0x55);

static struct bt_uuid_128 rx_uuid = BT_UUID_INIT_128(       // for Notify
	0x85, 0xb4, 0x93, 0x81, 0x58, 0x40, 0x0c, 0xbe,
	0xd1, 0x4d, 0x10, 0x00, 0xc4, 0x55, 0x00, 0x55);

static struct bt_uuid_128 tx_uuid = BT_UUID_INIT_128(       // for Write
	0x85, 0xb4, 0x93, 0x81, 0x58, 0x40, 0x0c, 0xbe,
	0xd1, 0x4d, 0x20, 0x00, 0xc4, 0x55, 0x00, 0x55);

static struct bt_conn *default_conn;
static uint16_t default_conn_handle;
static uint16_t tx_handle;
static struct bt_gatt_discover_params discover_params;

// --------------------------------------------------
// Notify callback for receiving data from peripheral
// --------------------------------------------------
static int rcvd_flag = 0;
static uint8_t notify_cb(struct bt_conn *conn,
                         struct bt_gatt_subscribe_params *params,
                         const void *data, uint16_t length) {
    if (!data) {
        LOG_ERR("8.1x Notification stopped\n");
        params->value_handle = 0U;
        return BT_GATT_ITER_STOP;
    }
    int ret = gpio_pin_set_dt(&led_builtin, true); k_msleep(5);

    // reveived data from peripheral
    memcpy(recv_buf, data, DATANUM);
    rcvd_flag = 1;
    
    // send data to peripheral
    int err = bt_gatt_write_without_response(conn, tx_handle, send_buf, DATANUM, false);
    if (err) {
        LOG_ERR("8.2x Write failed %d", err);
    }

    ret = gpio_pin_set_dt(&led_builtin, false);     
    return BT_GATT_ITER_CONTINUE;
}

// ------------------------------------------------
// Discover characteristics and subscribe to Notify
// ------------------------------------------------
static struct bt_gatt_subscribe_params sub_params;
static uint16_t rx_handle = 0;
static uint16_t tx_handle = 0;
static bool subscribed = false;

static uint8_t discover_char_func(struct bt_conn *conn,
                             const struct bt_gatt_attr *attr,
                             struct bt_gatt_discover_params *params) {
    if (!attr) {
        LOG_INF("7.3 Discover complete");
        memset(params, 0, sizeof(*params));

        // If 2 characteristics are found, subscribe
        if (!subscribed && rx_handle != 0 && tx_handle != 0) {
            sub_params.notify = notify_cb;
            sub_params.value  = BT_GATT_CCC_NOTIFY;
            sub_params.value_handle = rx_handle;
            sub_params.ccc_handle   = rx_handle + 1;

            int err = bt_gatt_subscribe(conn, &sub_params);
            LOG_INF("7.4 Subscribed after discovery: %d", err);
            subscribed = true;
        }
        return BT_GATT_ITER_STOP;
    }

    struct bt_gatt_chrc *chrc = (struct bt_gatt_chrc *)attr->user_data;

    if (!bt_uuid_cmp(chrc->uuid, &rx_uuid.uuid)) {
        rx_handle = chrc->value_handle;
        LOG_INF("7.1 Found RX characteristic (Notify), handle 0x%04x", rx_handle);

    } else if (!bt_uuid_cmp(chrc->uuid, &tx_uuid.uuid)) {
        tx_handle = chrc->value_handle;
        LOG_INF("7.2 Found TX characteristic (Write), handle 0x%04x", tx_handle);
    }
    return BT_GATT_ITER_CONTINUE;
}

// --------------------------------------------------------
// Service discovered, initiating characteristics discovery 
// --------------------------------------------------------
static uint8_t discover_service_func(struct bt_conn *conn,
                                     const struct bt_gatt_attr *attr,
                                     struct bt_gatt_discover_params *params) {
    if (!attr) {
        LOG_INF("5.3 Service discovery complete");
        memset(params, 0, sizeof(*params));
        discover_params.uuid = NULL;
        discover_params.func = discover_char_func;
        discover_params.start_handle = 0x0001;
        discover_params.end_handle = 0xffff;
        discover_params.type = BT_GATT_DISCOVER_CHARACTERISTIC;

        LOG_INF("6.0 Discovering characteristics");
        int rc = bt_gatt_discover(default_conn, &discover_params);
        if (rc) {
            LOG_ERR("6.1x Characteristic discovery failed %d", rc);
        }
        return BT_GATT_ITER_STOP;
    }
    LOG_INF("5.2 Service found: handle 0x%04x", attr->handle);
    return BT_GATT_ITER_CONTINUE;
}

// -----------------------------------------------------
// Connection established, so starting service discovery
// -----------------------------------------------------
static void connected(struct bt_conn *conn, uint8_t conn_err) {
    if (conn_err) {
        LOG_ERR("4.2x Connection failed %u\n", conn_err);
        default_conn = NULL;
        return;
    }
	default_conn = bt_conn_ref(conn);
	int err = bt_hci_get_conn_handle(default_conn, &default_conn_handle);
	if (err) {
		LOG_ERR("No connection handle (err %d)\n", err);
	}
    LOG_INF("4.2 Connected\n");

    // starting service discovery
    discover_params.uuid = &service_uuid.uuid;
    discover_params.func = discover_service_func;
    discover_params.start_handle = 0x0001;
    discover_params.end_handle = 0xffff;
    discover_params.type = BT_GATT_DISCOVER_PRIMARY;

    err = bt_gatt_discover(conn, &discover_params);
    if (err) {
        LOG_ERR("5.1x Service discovery failed %d\n", err);
    }
}

// ---------------------
// Disconnected callback
// ---------------------
static void disconnected(struct bt_conn *conn, uint8_t reason) {
    LOG_INF("Disconnected (reason %u)\n", reason);
    default_conn = NULL;
}

// -------------------------------
// Connection callbacks definitiom
// -------------------------------
BT_CONN_CB_DEFINE(conn_callbacks) = {
    .connected = connected,
    .disconnected = disconnected,
};

// --------------------------------------------
// Scan to find a peripheral with a target UUID
// --------------------------------------------
static bool adv_has_target_uuid(struct net_buf_simple *ad) {
    for (size_t i = 0; i < ad->len;) {
        uint8_t len = ad->data[i++];
        if (len == 0) break;
        uint8_t type = ad->data[i++];
        if (type == BT_DATA_UUID128_ALL || type == BT_DATA_UUID128_SOME) {
            if (memcmp(&ad->data[i], &service_uuid.val, 16) == 0) {
                return true;
            }
        }
        i += len - 1;
    }
    return false;
}

// --------------------------------------------------------
// If the target is found, stop scanning and try to connect
// --------------------------------------------------------
static void device_found(const bt_addr_le_t *addr, int8_t rssi,
                         uint8_t adv_type, struct net_buf_simple *ad) {
    char addr_str[BT_ADDR_LE_STR_LEN];
    bt_addr_le_to_str(addr, addr_str, sizeof(addr_str));
    if (!adv_has_target_uuid(ad)) return;

    LOG_INF("3.1 UUID match! Stopping scan...\n");
    int err = bt_le_scan_stop();
    if (err) {
        LOG_ERR("3.2x bt_le_scan_stop failed %d\n", err);
    } else {
        LOG_INF("3.2 bt_le_scan_stop done\n");
    }

    struct bt_le_conn_param *param = BT_LE_CONN_PARAM_DEFAULT;
    err = bt_conn_le_create(addr, BT_CONN_LE_CREATE_CONN, param, &default_conn);
    if (err) {
        LOG_ERR("4.1x bt_conn_le_create failed: %d\n", err);
        return;
    } else {
        LOG_INF("4.1 Connecting...\n");
    }
}

// --------------------------------------------
// Get the current RSSI value during connection
// --------------------------------------------
static void read_conn_rssi(uint16_t handle, int8_t *rssi) {
	struct net_buf *buf, *rsp = NULL;
	struct bt_hci_cp_read_rssi *cp;
	struct bt_hci_rp_read_rssi *rp;
	int err;

	buf = bt_hci_cmd_alloc(K_FOREVER);
	if (!buf) {
		LOG_ERR("Unable to allocate command buffer\n");
		return;
	}

	cp = net_buf_add(buf, sizeof(*cp));
	cp->handle = sys_cpu_to_le16(handle);

	err = bt_hci_cmd_send_sync(BT_HCI_OP_READ_RSSI, buf, &rsp);
	if (err) {
		LOG_ERR("Read RSSI err: %d\n", err);
		return;
	}

	rp = (void *)rsp->data;
	*rssi = rp->rssi;
	net_buf_unref(rsp);
}

// *********************************************************************************************
//                                              main
// *********************************************************************************************
int main(void)
{
    LOG_INF("XIAO_nRF54L15 Central\n");
    int err;
    default_conn = NULL;

	// ports and on board devices initialization
	err = port_init();
	if (err < 0) {
		LOG_ERR("0.1x port_init err:%d\n", err);
	}
	LOG_INF("0.1 port_init done\n");

    // vbat read adc initialization
	err = adc_init();
	if (err < 0) {
		LOG_ERR("0.2x adc_init err:%d\n", err);
	}
	LOG_INF("0.2 adc_init done\n");

	// SSD1306 with u8g2 initialization
	err = ssd_init();
	if (err < 0) {
		LOG_ERR("0.3x ssd_init err:%d\n", err);
	}
	LOG_INF("0.3 ssd_init done\n");

	// bluetooth initialization
    err = bt_enable(NULL);
    if (err) {
        LOG_ERR("1.1x bt_enable failed: %d\n", err);
        return err;
    }
    LOG_INF("1.2 Initialization done.\n");

    // Scanning
    struct bt_le_scan_param scan_param = {
        .type = BT_LE_SCAN_TYPE_ACTIVE,
        .interval = BT_GAP_SCAN_FAST_INTERVAL,
        .window = BT_GAP_SCAN_FAST_WINDOW,
    };
    err = bt_le_scan_start(&scan_param, device_found);
    if (err) {
        LOG_ERR("2.1x bt_le_scan_start failed: %d\n", err);
        return err;
    }
    LOG_INF("2.1 bt_le_scan_start done\n");    

    // loop
    while(1) {
        if (rcvd_flag == 1) {
            uint32_t timestamp = k_uptime_get_32();

        	// Vbat read
			err = adc_read_dt(&adc_channels[7], &sequence);
			if (err < 0) {
				LOG_ERR("Could not read adc (%d)\n", err);
				return 0;
			}
			val_mv = (int32_t)adc_buf;

			// conversion to mV
			err = adc_raw_to_millivolts_dt(&adc_channels[7], &val_mv);
			if (err < 0) {
				LOG_ERR("value in mV not available\n");
			}
			uint16_t vbat = (uint16_t)(val_mv * 2);
        
            // received data buffering
            memcpy(ud.dataBuff8, recv_buf, DATANUM);
            rcvd_flag = 0;
            uint32_t rx_time = ud.dataBuff32[0];
            uint32_t rx_rand = ud.dataBuff32[2];
            uint16_t rx_vbat = ud.dataBuff16[6];
            int8_t rx_rssi = ud.dataBuff16[7];

            LOG_INF("Rcvd %8u %08X %4u %3d", rx_time, rx_rand, rx_vbat, rx_rssi);

            // send data buffering
            uint32_t rand = sys_rand32_get();	// data sample
            int8_t rssi = -127;
            read_conn_rssi(default_conn_handle, &rssi);     // peripheral rssi
            ud.dataBuff32[0] = timestamp;
            ud.dataBuff32[2] = rand;
            ud.dataBuff16[6] = vbat;
            ud.dataBuff16[7] = (uint16_t)rssi;
            memcpy(send_buf, ud.dataBuff8, DATANUM);

            LOG_INF("Sent %8u %08X %4u %3d\n", timestamp, rand, vbat, rssi);

            // data display
            char line_1_str[32];
            char line_2_str[32];
            char line_3_str[32];
            char line_4_str[32];

            snprintf(line_1_str, sizeof(line_1_str), "Rcvd %08X", rx_rand);
            snprintf(line_2_str, sizeof(line_2_str), "%4dmV %3ddBm", rx_vbat, rx_rssi);
            snprintf(line_3_str, sizeof(line_3_str), "Sent %08X", rand);
            snprintf(line_4_str, sizeof(line_4_str), "%4dmV %3ddBm", vbat, rssi);

            lv_label_set_text(line_1, line_1_str);
            lv_label_set_text(line_2, line_2_str);
            lv_label_set_text(line_3, line_3_str);
            lv_label_set_text(line_4, line_4_str);
	        lv_task_handler();
            // 36 mS
        }
        k_msleep(50);
    }    
}



