#include <Arduino.h>
#include <nrf_timer.h>
#include <Adafruit_SPIFlash.h> 

const unsigned long kSerialSpeed = 115200;

// duration of count of anemometer pulses
const u_int32_t kSleepTime = 3;    
// pin to receive pulses from anemometer    
const u_int8_t kInterruptPin = D6;

const u_int32_t kPinBatEnable = 14; 
const u_int32_t kPinBatADC = PIN_VBAT;

// normally high, pulses low once every revolution
volatile bool flag_anemometer_pulse = false;
// wakes every few seconds to count pulses for windspeed
volatile bool flag_timer = false;  

// Flash Class instance
Adafruit_FlashTransport_QSPI flashTransport;

// semaphore used to halt processor and sleep
SemaphoreHandle_t xSemaphore;

// Initialise low frequency RTC to save power
// receives time duration in seconds / 1024
void InitialiseLFRTC(unsigned long RTCcount) 
{
  NRF_CLOCK->TASKS_LFCLKSTOP = 1;
  NRF_CLOCK->LFCLKSRC = 1;       // select LFXO
  NRF_CLOCK->TASKS_LFCLKSTART = 1;
  while(NRF_CLOCK->LFCLKSTAT != 0x10001)
  
  NRF_RTC2->TASKS_STOP = 1;      // stop counter 
  NRF_RTC2->TASKS_CLEAR = 1;     // clear counter
  NRF_RTC2->PRESCALER = 31;      // set counter prescaler, fout=32768/(1+PRESCALER)　1024Hz
  NRF_RTC2->CC[0] = RTCcount;    // set value for TRC compare register 0
  NRF_RTC2->INTENSET = 0x10000;  // enable interrupt CC[0] compare match event
  NRF_RTC2->EVTENCLR = 0x10000;  // clear counter when CC[0] event
  NVIC_SetPriority(RTC2_IRQn,3); // Higher priority than SoftDeviceAPI
  NVIC_EnableIRQ(RTC2_IRQn);     // enable interrupt  
  NRF_RTC2->TASKS_START = 1;     // start Timer
}

void PowerDownFlash(void) {
  flashTransport.begin();
  flashTransport.runCommand(0xB9); 
  delayMicroseconds(5);
  flashTransport.end();  
}

// interrupt service routine for RTC timer
// NB extern to avoid C++ name mangling
extern "C" void RTC2_IRQHandler(void)
{
  if ((NRF_RTC2->EVENTS_COMPARE[0] != 0) && ((NRF_RTC2->INTENSET & 0x10000) != 0)) {
    NRF_RTC2->EVENTS_COMPARE[0] = 0;  // clear compare register 0 event
    NRF_RTC2->TASKS_CLEAR = 1;        // clear counter

    flag_timer = true;   
  }
  xSemaphoreGive(xSemaphore);
}

// interrupt service routine for anemometer pulse
void ISRDetectPulse() {
  flag_anemometer_pulse = true;
  xSemaphoreGive(xSemaphore);
}

// NB required to prevent over-voltage of ADC pin
void EnableVoltageDivider(void) {
  pinMode(kPinBatEnable, OUTPUT);
  digitalWrite(kPinBatEnable, LOW);
}

void ConfigureLEDPins(void){
  // make sure led control pins configured for OUTPUT 
  // and set HIGH so that leds are off
  pinMode(LED_RED, OUTPUT);
  pinMode(LED_BLUE, OUTPUT);
  pinMode(LED_GREEN, OUTPUT);
  digitalWrite(LED_RED, HIGH);
  digitalWrite(LED_BLUE, HIGH);
  digitalWrite(LED_GREEN, HIGH);
}

void setup() {
  // necessary to prevent over-voltage of ADC pin
  // increases quiescent current by Vbat / 1.5E6
  // for 4.5 V supply : 3 microamps
  EnableVoltageDivider();
  ConfigureLEDPins();

  // enable interrupt on anemometer pulses
  // normally HIGH, goes LOW once per revolution
  // TMR sensor so no debounce required
  pinMode(kInterruptPin, INPUT);
  attachInterrupt(digitalPinToInterrupt(kInterruptPin), ISRDetectPulse, FALLING);

  // Use DC-DC converter to save power - best improvement is on high duty cycle
  NRF_POWER->DCDCEN = 1;   

  PowerDownFlash();
 
  InitialiseLFRTC(1024 * kSleepTime);

  xSemaphore = xSemaphoreCreateBinary();
}

void loop() {
  xSemaphoreTake(xSemaphore, portMAX_DELAY);

  if (flag_timer == true) {
    flag_timer = false;

    // flash green to indicate RTC interrupt processed
    digitalWrite(LED_GREEN, LOW);
    delay(10);       
    digitalWrite(LED_GREEN, HIGH);  
  }
  else if (flag_anemometer_pulse == true) {
    flag_anemometer_pulse = false;

    // flash blue to indicate anemometer interrupt processed
    digitalWrite(LED_BLUE, LOW);
    delay(10);       
    digitalWrite(LED_BLUE, HIGH); 
  }
}
