/*
 * XIAO nRF54L15 — VBAT → BAS % + RAW mV (custom) + LED blink
 * Rev: 2.2 — 2025-11-01
 */
#include <inttypes.h>
#include <stddef.h>
#include <stdint.h>
#include <string.h>

#include <zephyr/kernel.h>
#include <zephyr/logging/log.h>
LOG_MODULE_REGISTER(app, LOG_LEVEL_INF);

#include <zephyr/device.h>
#include <zephyr/devicetree.h>
#include <zephyr/drivers/gpio.h>
#include <zephyr/drivers/adc.h>
#include <zephyr/drivers/regulator.h>

#include <zephyr/settings/settings.h>
#include <zephyr/bluetooth/bluetooth.h>
#include <zephyr/bluetooth/gatt.h>
#include <zephyr/bluetooth/services/bas.h>
#include <zephyr/sys/byteorder.h>

/* ---------- LED (blink to show life without USB) ---------- */
#define LED0_NODE DT_ALIAS(led0)
static const struct gpio_dt_spec led = GPIO_DT_SPEC_GET(LED0_NODE, gpios);

static void blink_fn(struct k_timer *t) { gpio_pin_toggle_dt(&led); }
K_TIMER_DEFINE(blink_timer, blink_fn, NULL);

/* ---------- ADC (VBAT) via zephyr,user/io-channels ---------- */
#if !DT_NODE_EXISTS(DT_PATH(zephyr_user)) || \
    !DT_NODE_HAS_PROP(DT_PATH(zephyr_user), io_channels)
#error "No suitable devicetree overlay: zephyr,user/io-channels missing"
#endif

#define ADC_SPEC_FROM_PROP(node_id, prop, idx) ADC_DT_SPEC_GET_BY_IDX(node_id, idx),
static const struct adc_dt_spec adc_channels[] = {
    DT_FOREACH_PROP_ELEM(DT_PATH(zephyr_user), io_channels, ADC_SPEC_FROM_PROP)
};

#if DT_NODE_HAS_STATUS(DT_NODELABEL(vbat_pwr), okay)
#define HAVE_VBAT_REG 1
static const struct device *const vbat_reg = DEVICE_DT_GET(DT_NODELABEL(vbat_pwr));
#else
#define HAVE_VBAT_REG 0
#endif

static uint16_t sample_buf;
static struct adc_sequence seq = {
    .buffer = &sample_buf,
    .buffer_size = sizeof(sample_buf),
};

static int vbat_mv_read(int32_t *out_mv)
{
#if HAVE_VBAT_REG
    if (device_is_ready(vbat_reg)) {
        regulator_enable(vbat_reg);
        k_msleep(50);
    }
#endif

    const struct adc_dt_spec *ch = &adc_channels[0];
    if (!device_is_ready(ch->dev)) {
        return -ENODEV;
    }

    int err = adc_channel_setup_dt(ch);
    if (err) goto done;

    adc_sequence_init_dt(ch, &seq);
    err = adc_read_dt(ch, &seq);
    if (err) goto done;

    int32_t mv = ch->channel_cfg.differential ?
                 (int32_t)((int16_t)sample_buf) : (int32_t)sample_buf;

    err = adc_raw_to_millivolts_dt(ch, &mv);
    if (err >= 0) {
        mv *= 2;              /* XIAO VBAT divider = 1/2 */
        *out_mv = mv;
        err = 0;
    } else {
        *out_mv = -1;
    }

done:
#if HAVE_VBAT_REG
    if (device_is_ready(vbat_reg)) {
        regulator_disable(vbat_reg);
    }
#endif
    return err;
}

/* Simple 1-cell LiPo curve (same as your console mapping) */
static int pct_from_mv(int32_t mv)
{
    if (mv < 3300) mv = 3300;
    if (mv > 4200) mv = 4200;
    return (int)((mv - 3300) * 100 / 900);
}

/* ---------- Custom GATT service: RAW millivolts (uint16_t LE) ---------- */
/* New, random 128-bit UUIDs just for this app */
#define BT_UUID_VBAT_SVC_VAL  BT_UUID_128_ENCODE(0x9c0a6ad3,0x3b8e,0x4b70,0x9c9e,0x9e4f8a2d0a10)
#define BT_UUID_VBAT_MV_VAL   BT_UUID_128_ENCODE(0x9c0a6ad3,0x3b8e,0x4b70,0x9c9e,0x9e4f8a2d0a11)

static struct bt_uuid_128 vbat_svc_uuid = BT_UUID_INIT_128(BT_UUID_VBAT_SVC_VAL);
static struct bt_uuid_128 vbat_mv_uuid  = BT_UUID_INIT_128(BT_UUID_VBAT_MV_VAL);

static uint16_t vbat_mv_value = 0;           /* host-endian mV */
static bool vbat_notify_enabled = false;

static ssize_t read_vbat_mv(struct bt_conn *conn, const struct bt_gatt_attr *attr,
                            void *buf, uint16_t len, uint16_t offset)
{
    uint16_t mv_le = sys_cpu_to_le16(vbat_mv_value);
    return bt_gatt_attr_read(conn, attr, buf, len, offset, &mv_le, sizeof(mv_le));
}

static void vbat_ccc_cfg_changed(const struct bt_gatt_attr *attr, uint16_t value)
{
    vbat_notify_enabled = (value == BT_GATT_CCC_NOTIFY);
}

/* Attribute order: [0]=Primary, [1]=Char Decl, [2]=Char Value, [3]=CCC */
BT_GATT_SERVICE_DEFINE(vbat_svc,
    BT_GATT_PRIMARY_SERVICE(&vbat_svc_uuid),
    BT_GATT_CHARACTERISTIC(&vbat_mv_uuid.uuid,
                           BT_GATT_CHRC_READ | BT_GATT_CHRC_NOTIFY,
                           BT_GATT_PERM_READ,
                           read_vbat_mv, NULL, NULL),
    BT_GATT_CCC(vbat_ccc_cfg_changed, BT_GATT_PERM_READ | BT_GATT_PERM_WRITE),
);

/* Helper to get pointer to the characteristic VALUE attribute (index 2) */
#define VBAT_VALUE_ATTR (&vbat_svc.attrs[2])

/* ---------- BLE init + advertising ---------- */
static void bt_ready(void)
{
    if (IS_ENABLED(CONFIG_SETTINGS)) {
        settings_load();
    }

    int err = bt_le_adv_start(BT_LE_ADV_CONN_NAME, NULL, 0, NULL, 0);
    if (err) {
        LOG_ERR("Advertising start failed (%d)", err);
    } else {
        LOG_INF("Advertising as: \"%s\"", CONFIG_BT_DEVICE_NAME);
    }
}

/* ---------- Periodic VBAT work: updates BAS % and RAW mV ---------- */
static void vbat_work_fn(struct k_work *w);
K_WORK_DELAYABLE_DEFINE(vbat_work, vbat_work_fn);

static void vbat_work_fn(struct k_work *w)
{
    int32_t mv = 0;
    if (vbat_mv_read(&mv) == 0 && mv > 0) {
        vbat_mv_value = (uint16_t)mv;

        int pct = pct_from_mv(mv);
        if (pct < 0)   pct = 0;
        if (pct > 100) pct = 100;

        /* 1) Update BAS (%). If CCC enabled, this will notify automatically. */
        bt_bas_set_battery_level((uint8_t)pct);

        /* 2) Notify RAW mV (uint16 LE) on our custom characteristic if enabled */
        if (vbat_notify_enabled) {
            uint16_t mv_le = sys_cpu_to_le16(vbat_mv_value);
            bt_gatt_notify(NULL, VBAT_VALUE_ATTR, &mv_le, sizeof(mv_le));
        }

        LOG_INF("VBAT=%" PRId32 " mV (%d%%)", mv, pct);
    } else {
        LOG_WRN("VBAT read failed");
    }

    k_work_schedule(&vbat_work, K_SECONDS(30));
}

/* ---------- main ---------- */
int main(void)
{
    LOG_INF("=== XIAO nRF54L15 VBAT → BAS/DIS + RAWmV — Rev 2.2 ===");
    LOG_INF("Board: %s | Build: %s %s", CONFIG_BOARD, __DATE__, __TIME__);

    /* LED first so we always get a heartbeat */
    if (device_is_ready(led.port)) {
        gpio_pin_configure_dt(&led, GPIO_OUTPUT_INACTIVE);
        k_timer_start(&blink_timer, K_MSEC(200), K_MSEC(500));
    }

    if (bt_enable(NULL)) {
        LOG_ERR("Bluetooth init failed");
        return 0;
    }
    bt_ready();

    /* Kick off first reading shortly after boot */
    k_work_schedule(&vbat_work, K_SECONDS(1));

    /* Idle forever; timers/work do the job */
    for (;;) {
        k_sleep(K_SECONDS(60));
    }
    return 0;
}
